<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Team;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class TeamsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $this->authorize('viewAny', WebsiteSetting::class);
        if ($request->ajax()) {
            // dd($request->all());
            $draw = $request->get('draw');
            $start = $request->get("start");
            $rowperpage = $request->get("length"); // Rows display per page
            $search_arr = $request->get('search');
            $searchValue = $search_arr['value']; // Search value
            $totalRecords = Team::select('count(*) as allcount')->count();
            $totalRecordswithFilter = Team::select('count(*) as allcount')
                ->where('name', 'like', '%' . $searchValue . '%')
                ->orWhere('name_ar', 'like', '%' . $searchValue . '%')
                ->count();

            $teams = Team::select('teams.*')
                ->where('name', 'like', '%' . $searchValue . '%')
                ->orWhere('name_ar', 'like', '%' . $searchValue . '%')
                ->skip($start)
                ->take($rowperpage)
                ->orderBy('teams.id', 'desc')
                ->get();


            $records = [];

            foreach($teams as $item)
            {
                $records[] = [
                    'record_select' =>view('admin.teams.datatable.record',[ 'item' => $item ])->render(),
                    'id' => $item->id,
                    'image' => $item->image ? '<img src="'. asset('storage/'.$item->image) .'" width="70">' : __('No image uploaded'),
                    'name' => \LaravelLocalization::getCurrentLocale() == 'ar' ? $item->name_ar : $item->name,
                    'position' => \LaravelLocalization::getCurrentLocale() == 'ar' ? $item->position_ar : $item->position,
                    'actions' => view('admin.teams.datatable.buttons',[ 'item' => $item ])->render() ,
                ];
            }

            $response = [
                "draw" => intval($draw),
                "iTotalRecords" => $totalRecords,
                "iTotalDisplayRecords" => $totalRecordswithFilter,
                "aaData" => $records
            ];

            return response()->json($response);

        }

        return view('admin.teams.index');
    }


    public function store(Request $request)
    {
        # code...
        $this->authorize('viewAny', WebsiteSetting::class);
        $request->validate([
            'image' => 'required|mimes:png,jpg,jpeg',
            'name' => 'required|string|max:255',
            'name_ar' => 'required|string|max:255',
            'position' => 'required|string|max:255',
            'position_ar' => 'required|string|max:255',
            'description' => 'nullable',
            'description_ar' => 'nullable',
        ]);

        $data = $request->all();
        $image = null;

        if($request->hasFile('image') && $request->file('image')->isValid()){
            $image = $request->file('image')->store('teams', 'public');
            $data['image'] = $image;
        }

        Team::create($data);

        return redirect()->route('c-panel.teams.index')->with('success', __('Added Done !'));
    }

    public function update(Request $request, Team $team)
    {
        # code...
        $this->authorize('viewAny', WebsiteSetting::class);
        $request->validate([
            'image' => 'nullable|mimes:png,jpg,jpeg',
            'name' => 'required|string|max:255',
            'name_ar' => 'required|string|max:255',
            'position' => 'required|string|max:255',
            'position_ar' => 'required|string|max:255',
            'description' => 'nullable',
            'description_ar' => 'nullable',
        ]);

        $data = $request->all();
        $image = $team->image;

        if($request->hasFile('image') && $request->file('image')->isValid()){
            if($team->image)
                Storage::disk('public')->delete($team->image);
            $image = $request->file('image')->store('teams', 'public');
            $data['image'] = $image;
        }

        $team->update($data);
        return redirect()->route('c-panel.teams.index')->with('success', __('Updated Done !'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Team $team)
    {
        // $this->authorize('delete', $department);
        $this->authorize('viewAny', WebsiteSetting::class);
        if($team->image)
            Storage::disk('public')->delete($team->image);
        $team->delete();
        return redirect()->route('c-panel.teams.index')->with('success',__('Deleted Done !'));
    }

    public function bulkDelete()
    {
        // dd(json_decode(request()->record_ids));
        $this->authorize('viewAny', WebsiteSetting::class);
        if(count(json_decode(request()->record_ids)) != 0){
            foreach (json_decode(request()->record_ids) as $recordId) {
                $team = Team::FindOrFail($recordId);
                // $this->authorize('delete', $team);
                $this->delete($team);
            }//end of for each
        }else{
            return redirect()->route('c-panel.teams.index')->with('error',__('Please Select Record Before Delete'));
        }

        return redirect()->route('c-panel.teams.index')->with('success',__('Selected  Deleted Done !'));
    }// end of bulkDelete

    private function delete(Team $team)
    {
        // $this->authorize('delete', $team);
        $this->authorize('viewAny', WebsiteSetting::class);
        if($team->image)
            Storage::disk('public')->delete($team->image);
        $team->delete();
    }
}
