<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rules;
use Illuminate\Support\Facades\Hash;

class CustomersController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('viewAnyCustomer', User::class);
        # code...
        if($request->search != '' || $request->department_id != ''){
            if($request->search && $request->department_id)
                $customers = User::where('role_id',0)->where([['name','LIKE',"%$request->search%"], ['department_id',$request->department_id]])->latest()->paginate(12);
            else if($request->search)
                $customers = User::where('role_id',0)->where('name','LIKE',"%$request->search%")->latest()->paginate(12);
            else if($request->department_id)
                $customers = User::where('role_id',0)->where('department_id',$request->department_id)->latest()->paginate(12);
        }else{
            $customers = User::where('role_id',0)->latest()->paginate(12);
        }

        return view('admin.customers.index',[
            'customers' => $customers
        ]);
    }

    public function show(User $customer)
    {
        # code...
        $this->authorize('viewAnyCustomer', User::class);
        return view('admin.customers.show',[
            'customer' => $customer,
        ]);
    }


    public function customerSubscriptions(Request $request, User $customer)
    {
        $this->authorize('viewAnyCustomer', User::class);
        if ($request->ajax()) {
            // dd($request->all());
            $draw = $request->get('draw');
            $start = $request->get("start");
            $rowperpage = $request->get("length"); // Rows display per page
            $search_arr = $request->get('search');
            $searchValue = $search_arr['value']; // Search value
            $totalRecords = $customer->subscriptions()->where('type', $request->type)->select('count(*) as allcount')->count();
            $totalRecordswithFilter = $customer->subscriptions()->where('type', $request->type)->select('count(*) as allcount')
                ->where(function ($q) use ($searchValue){
                    $q->whereHas('customer', function ($query) use ($searchValue){
                        $query->where('name','LIKE',"%$searchValue%");
                    })
                    ->orWhereHas('domin', function ($query) use ($searchValue){
                        $query->where('name','LIKE',"%$searchValue%");
                    })
                    ->orWhereHas('plan', function ($query) use ($searchValue){
                        $query->whereHas('group', function ($query) use ($searchValue){
                            $query->where('name','LIKE',"%$searchValue%");
                        });
                    })
                    ->orWhere('id', 'like', '%' . $searchValue . '%');
                })
                ->count();

            $subscriptions = $customer->subscriptions()->where('type', $request->type)->select('subscriptions.*')
                ->where(function ($q) use ($searchValue){
                    $q->whereHas('customer', function ($query) use ($searchValue){
                        $query->where('name','LIKE',"%$searchValue%");
                    })
                    ->orWhereHas('domin', function ($query) use ($searchValue){
                        $query->where('name','LIKE',"%$searchValue%");
                    })
                    ->orWhereHas('plan', function ($query) use ($searchValue){
                        $query->whereHas('group', function ($query) use ($searchValue){
                            $query->where('name','LIKE',"%$searchValue%");
                        });
                    })
                    ->orWhere('id', 'like', '%' . $searchValue . '%');
                })
                ->skip($start)
                ->take($rowperpage)
                ->orderBy('subscriptions.id', 'desc')
                ->get();


            $records = [];

            foreach($subscriptions as $item)
            {
                $status = '';
                if($item->status == 0) {
                    $status = __('Active');
                }
                else if($item->status == 1) {
                    $status = __('Canceled');
                }
                else if($item->status == 2) {
                    $status = __('Stopped');
                }
                else if($item->status == 3) {
                    $status = __('Not Enabled');
                }

                $status_0 = '';
                if($item->status_0 == 0) {
                    $status_0 = __('Paid');
                }
                else if($item->status_0 == 1) {
                    $status_0 = __('Worthy');
                }

                $currancy = '';
                if($item->currancy == 0) {
                    $currancy = '$';
                } else if ($item->currancy == 1) {
                    $currancy = 'SAR';
                } else if ($item->currancy == 2) {
                    $currancy = 'ILS';
                }

                $records[] = [
                    'id' => $item->id,
                    'name' => $item->type == 0 ? $item->plan->group->name . ' ' . $item->plan->title : $item->domin->name,
                    'start_date' => $item->start_date,
                    'subscrip_date' => $item->subscrip_date,
                    'due_date' => $item->due_date,
                    'end_date' => $item->end_date,
                    'price' => $item->price . ' ' . $currancy,
                    'status' => $status,
                    'status_0' => $status_0,
                ];
            }

            $response = [
                "draw" => intval($draw),
                "iTotalRecords" => $totalRecords,
                "iTotalDisplayRecords" => $totalRecordswithFilter,
                "aaData" => $records
            ];

            return response()->json($response);

        }
    }

    public function customerBills(Request $request, User $customer)
    {
        $this->authorize('viewAnyCustomer', User::class);
        if ($request->ajax()) {
            // dd($request->all());
            $draw = $request->get('draw');
            $start = $request->get("start");
            $rowperpage = $request->get("length"); // Rows display per page
            $search_arr = $request->get('search');
            $searchValue = $search_arr['value']; // Search value
            $totalRecords = $customer->bills()->where('status', 1)->select('count(*) as allcount')->count();
            $totalRecordswithFilter = $customer->bills()->where('status', 1)->select('count(*) as allcount')
                ->where('id', 'like', '%' . $searchValue . '%')
                ->count();

            $bills = $customer->bills()->where('status', 1)->select('bills.*')
                ->where('id', 'like', '%' . $searchValue . '%')
                ->skip($start)
                ->take($rowperpage)
                ->orderBy('bills.id', 'desc')
                ->get();


            $records = [];

            foreach($bills as $item)
            {
                $status = '';
                // if($item->status == 0){
                //     $status = '<span class="badge badge-soft-primary text-uppercase">' . __('New Offer') . '</span>';
                // }else if($item->status == 1){
                //     $status = '<span class="badge badge-soft-info text-uppercase">' . __('Under Review') . '</span>';
                // }else if($item->status == 2){
                //     $status = '<span class="badge badge-soft-warning text-uppercase">' . __('Under Update') . '</span>';
                // }else if($item->status == 3){
                //     $status = '<span class="badge badge-soft-success text-uppercase">' . __('Accepted') . '</span>';
                // }else if($item->status == 4){
                //     $status = '<span class="badge badge-soft-danger text-uppercase">' . __('Rejected') . '</span>';
                // }else if($item->status == 5){
                //     $status = '<span class="badge badge-soft-danger text-uppercase">' . __('Canceled') . '</span>';
                // }else if($item->status == 6){
                //     $status = '<span class="badge badge-soft-success text-uppercase">' . __('Converted to Invoice') . '</span>';
                // }

                if($item->status == 1){
                    $status = '<span class="badge badge-soft-danger text-uppercase">' . __('مستحقة') . '</span>';
                }

                if($item->currancy == 0) {
                    $currancy = '$';
                } else if ($item->currancy == 1) {
                    $currancy = 'SAR';
                } else if ($item->currancy == 2) {
                    $currancy = 'ILS';
                }

                $records[] = [
                    'id' => $item->id,
                    'date' => $item->date,
                    'end_date' => $item->end_date ?? __('Not Entered'),
                    'status' => $status,
                    'totel' => $item->totel . ' ' . $currancy,
                    'actions' => view('admin.bills.datatable.my-button',[ 'item' => $item ])->render() ,
                ];
            }

            $response = [
                "draw" => intval($draw),
                "iTotalRecords" => $totalRecords,
                "iTotalDisplayRecords" => $totalRecordswithFilter,
                "aaData" => $records
            ];

            return response()->json($response);

        }

        return view('admin.bills.my-bills');
    }

    public function store(Request $request)
    {
        # code...
        $this->authorize('createCustomer', User::class);
        $request->validate([
            'image' => 'nullable|mimes:png,jpg',
            'name' => 'required|string|max:255',
            'email' => 'email|required|unique:users',
            'password' => ['required','confirmed', Rules\Password::defaults()],
            'department_id' => 'required|exists:departments,id',
            'association' => 'required|string|max:255',
            'phone' => 'required|string|max:255',
            'country' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:255',
            'website' => 'nullable|string|max:255',
        ]);

        $data = $request->except(['image', 'password']);
        $data['role_id'] = 0;
        $data['password'] = Hash::make($request->password);

        $image = null;

        if($request->hasFile('image') && $request->file('image')->isValid())
        {
            $image = $request->file('image')->store('customers', 'public');
            $data['image'] = $image;
        }

        User::create($data);

        return redirect()->back()->with('success', __('Added Done !'));
    }

    public function update(Request $request, User $customer)
    {
        # code...
        $this->authorize('updateCustomer', $customer);
        $request->validate([
            'image' => 'nullable|mimes:png,jpg',
            'name' => 'required|string|max:255',
            'email' => 'email|required|unique:users,email,' . $customer->id,
            'password' => ['nullable','confirmed', Rules\Password::defaults()],
            'department_id' => 'required|exists:departments,id',
            'association' => 'required|string|max:255',
            'phone' => 'required|string|max:255',
            'country' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:255',
            'website' => 'nullable|string|max:255',
            'status' => 'nullable',
        ]);

        $data = $request->except(['image', 'password']);
        $data['role_id'] = 0;
        if($request->password)
            $data['password'] = Hash::make($request->password);

        $image = $customer->image;

        if($request->hasFile('image') && $request->file('image')->isValid())
        {
            if($image)
                Storage::disk('public')->delete($image);
            $image = $request->file('image')->store('customers', 'public');
            $data['image'] = $image;
        }

        $customer->update($data);

        return redirect()->back()->with('success', __('Updated Done !'));
    }

    public function destroy(User $customer)
    {
        # code...
        $this->authorize('deleteCustomer', $customer);
        if($customer->image)
            Storage::disk('public')->delete($customer->image);
        $customer->delete();
        return redirect()->route('c-panel.customers.index')->with('success', __('Deleted Done !'));

    }
}
